#include "common.h"

RawImageData loadBMPRaw(const char* filename) {
	FILE* file = fopen(filename, "rb");
	if(!file) {
		std::cout << "[ERROR] File not found: " << filename << std::endl;
		return {nullptr, 0, 0};
	}

	unsigned char header[54];
	fread(header, 1, 54, file);

	unsigned int dataPos = *(int*)&(header[0x0A]);
	int width            = *(int*)&(header[0x12]);
	int height           = *(int*)&(header[0x16]);
	unsigned short bits  = *(unsigned short*)&(header[0x1C]);

	unsigned char palette[1024];
	if(bits == 8) {
		fseek(file, 54, SEEK_SET);
		fread(palette, 1, 1024, file);
	}

	fseek(file, dataPos, SEEK_SET);
	int bytesPerPixel = (bits >= 24) ? (bits / 8) : 1;
	int rowSize = (width * bytesPerPixel + 3) & ~3;
	int imageSize = rowSize * abs(height);
	unsigned char* rawData = new unsigned char[imageSize];
	fread(rawData, 1, imageSize, file);
	fclose(file);

	unsigned char* destData = new unsigned char[width * height * 4];
	for(int y = 0; y < height; y ++) {
		for(int x = 0; x < width; x ++) {
			int srcIdx = y * rowSize + x * bytesPerPixel;
			int dstIdx = (y * width + x) * 4;

			if(bits == 8) {
				int colorIdx = rawData[srcIdx];
				destData[dstIdx + 0] = palette[colorIdx * 4 + 2];
				destData[dstIdx + 1] = palette[colorIdx * 4 + 1];
				destData[dstIdx + 2] = palette[colorIdx * 4 + 0];
				destData[dstIdx + 3] = 255;
			} else if(bits == 24) {
				destData[dstIdx + 0] = rawData[srcIdx + 2];
				destData[dstIdx + 1] = rawData[srcIdx + 1];
				destData[dstIdx + 2] = rawData[srcIdx + 0];
				destData[dstIdx + 3] = 255;
			} else if(bits == 32) {
				destData[dstIdx + 0] = rawData[srcIdx + 2];
				destData[dstIdx + 1] = rawData[srcIdx + 1];
				destData[dstIdx + 2] = rawData[srcIdx + 0];
				destData[dstIdx + 3] = rawData[srcIdx + 3];
			}
		}
	}

	delete[] rawData;
	return {destData, width, height};
}

UVRect getUV(int slotX, int slotY) {
	float uUnit = 1.0f / (float)ATLAS_SLOTS_W;
	float vUnit = 1.0f / (float)ATLAS_SLOTS_H;

	return {
		(float)slotX * uUnit, 
		(float)slotY * vUnit,
		(float)(slotX + 1) * uUnit, 
		(float)(slotY + 1) * vUnit
	};
}

void initTextureAtlas(GLuint& terrainAtlasID) {
	glGenTextures(1, &terrainAtlasID);
	glBindTexture(GL_TEXTURE_2D, terrainAtlasID);

	int fullSizeW = ATLAS_SLOTS_W * TEXTURE_SIZE;
	int fullSizeH = ATLAS_SLOTS_H * TEXTURE_SIZE;
	std::vector<unsigned char> emptyData(fullSizeW * fullSizeH * 4, 0); 
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, fullSizeW, fullSizeH, 0, GL_RGBA, GL_UNSIGNED_BYTE, emptyData.data());

	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
}

UVRect addFileToAtlas(const char* filename, int& nowSlotNumber, GLuint& terrainAtlasID) {
	int slotX = nowSlotNumber % ATLAS_SLOTS_W;
	int slotY = nowSlotNumber / ATLAS_SLOTS_W;
	RawImageData img = loadBMPRaw(filename);
	if(img.data) {
		glBindTexture(GL_TEXTURE_2D, terrainAtlasID);
		
		glTexSubImage2D(
			GL_TEXTURE_2D,
			0, 
			slotX * TEXTURE_SIZE,
			slotY * TEXTURE_SIZE, 
			img.width,
			img.height, 
			GL_RGBA,
			GL_UNSIGNED_BYTE,
			img.data
		);
		
		delete[] img.data;
		std::cout << "[ATLAS] Added: " << filename << " to slot (" << slotX << "," << slotY << ")" << std::endl;
		nowSlotNumber ++;
		return getUV(slotX, slotY);
	}
	
	return getUV(0, 0);
}

Skin::Skin(
	int dtype = 0,
	UVRect t = getUV(0, 0),
	UVRect s = getUV(0, 0),
	UVRect b = getUV(0, 0)
)  : top(t), side(s), bottom(b) {
	switch(dtype) {
		case 0:
			side = top;
			bottom = top;
			break;
		case 1:
			bottom = top;
			break;
		case 2:
			break;
		default:
			break;
	}
	
}

Block::Block(
	const std::string& name,
	const Skin& skin,
	bool isCollidable = true,
	bool isDisplayed = true,
	bool isOpaque = true
) : name(name), skin(skin), isCollidable(isCollidable), isDisplayed(isDisplayed), isOpaque(isOpaque) {}

BlockRegistry::BlockRegistry(int& nowSlotNumber, GLuint& terrainAtlasID) {
	initTextureAtlas(terrainAtlasID);
	UVRect uv_nf_t = addFileToAtlas("img/nf/top.bmp", nowSlotNumber, terrainAtlasID);
	
	UVRect uv_soil_t = addFileToAtlas("img/soil/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_grass_t = addFileToAtlas("img/grass/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_grass_s = addFileToAtlas("img/grass/side.bmp", nowSlotNumber, terrainAtlasID);
	
	UVRect uv_stone_t = addFileToAtlas("img/stone/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_cobblestone_t = addFileToAtlas("img/cobblestone/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_deepslate_t = addFileToAtlas("img/deepslate/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_obsidian_t = addFileToAtlas("img/obsidian/top.bmp", nowSlotNumber, terrainAtlasID);
	
	UVRect uv_granite_t = addFileToAtlas("img/granite/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_diorite_t = addFileToAtlas("img/diorite/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_gabbro_t = addFileToAtlas("img/gabbro/top.bmp", nowSlotNumber, terrainAtlasID);	
	UVRect uv_rhyolite_t = addFileToAtlas("img/rhyolite/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_basalt_t = addFileToAtlas("img/basalt/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_basalt_s = addFileToAtlas("img/basalt/side.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_andesite_t = addFileToAtlas("img/andesite/top.bmp", nowSlotNumber, terrainAtlasID);
	
	UVRect uv_sand_t = addFileToAtlas("img/sand/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_sandstone_t = addFileToAtlas("img/sandstone/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_mud_t = addFileToAtlas("img/mud/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_mudstone_t = addFileToAtlas("img/mudstone/top.bmp", nowSlotNumber, terrainAtlasID);
	
	UVRect uv_iron_t = addFileToAtlas("img/iron/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_copper_t = addFileToAtlas("img/copper/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_gold_t = addFileToAtlas("img/gold/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_tin_t = addFileToAtlas("img/tin/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_platinum_t = addFileToAtlas("img/platinum/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_uranium_t = addFileToAtlas("img/uranium/top.bmp", nowSlotNumber, terrainAtlasID);
	
	UVRect uv_water_t = addFileToAtlas("img/water/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_lava_t = addFileToAtlas("img/lava/top.bmp", nowSlotNumber, terrainAtlasID);
	
	UVRect uv_brick_t = addFileToAtlas("img/brick/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_brick_s = addFileToAtlas("img/brick/side.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_brick_b = addFileToAtlas("img/brick/bottom.bmp", nowSlotNumber, terrainAtlasID);
	
	UVRect uv_oak_log_t = addFileToAtlas("img/oak_log/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_oak_log_s = addFileToAtlas("img/oak_log/side.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_spruce_log_t = addFileToAtlas("img/spruce_log/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_spruce_log_s = addFileToAtlas("img/spruce_log/side.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_birch_log_t = addFileToAtlas("img/birch_log/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_birch_log_s = addFileToAtlas("img/birch_log/side.bmp", nowSlotNumber, terrainAtlasID);

	UVRect uv_leaf_t = addFileToAtlas("img/leaf/top.bmp", nowSlotNumber, terrainAtlasID);
	
	UVRect uv_oak_wood_t = addFileToAtlas("img/oak_wood/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_spruce_wood_t = addFileToAtlas("img/spruce_wood/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_birch_wood_t = addFileToAtlas("img/birch_wood/top.bmp", nowSlotNumber, terrainAtlasID);
	
	UVRect uv_wool_white_t = addFileToAtlas("img/wool_white/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_wool_lightgray_t = addFileToAtlas("img/wool_lightgray/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_wool_gray_t = addFileToAtlas("img/wool_gray/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_wool_black_t = addFileToAtlas("img/wool_black/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_wool_brown_t = addFileToAtlas("img/wool_brown/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_wool_red_t = addFileToAtlas("img/wool_red/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_wool_orange_t = addFileToAtlas("img/wool_orange/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_wool_yellow_t = addFileToAtlas("img/wool_yellow/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_wool_lime_t = addFileToAtlas("img/wool_lime/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_wool_green_t = addFileToAtlas("img/wool_green/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_wool_cyan_t = addFileToAtlas("img/wool_cyan/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_wool_skyblue_t = addFileToAtlas("img/wool_skyblue/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_wool_blue_t = addFileToAtlas("img/wool_blue/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_wool_magenta_t = addFileToAtlas("img/wool_magenta/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_wool_purple_t = addFileToAtlas("img/wool_purple/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_wool_pink_t = addFileToAtlas("img/wool_pink/top.bmp", nowSlotNumber, terrainAtlasID);
	
	UVRect uv_concrete_white_t = addFileToAtlas("img/concrete_white/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_concrete_lightgray_t = addFileToAtlas("img/concrete_lightgray/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_concrete_gray_t = addFileToAtlas("img/concrete_gray/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_concrete_black_t = addFileToAtlas("img/concrete_black/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_concrete_brown_t = addFileToAtlas("img/concrete_brown/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_concrete_red_t = addFileToAtlas("img/concrete_red/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_concrete_orange_t = addFileToAtlas("img/concrete_orange/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_concrete_yellow_t = addFileToAtlas("img/concrete_yellow/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_concrete_lime_t = addFileToAtlas("img/concrete_lime/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_concrete_green_t = addFileToAtlas("img/concrete_green/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_concrete_cyan_t = addFileToAtlas("img/concrete_cyan/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_concrete_skyblue_t = addFileToAtlas("img/concrete_skyblue/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_concrete_blue_t = addFileToAtlas("img/concrete_blue/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_concrete_magenta_t = addFileToAtlas("img/concrete_magenta/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_concrete_purple_t = addFileToAtlas("img/concrete_purple/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_concrete_pink_t = addFileToAtlas("img/concrete_pink/top.bmp", nowSlotNumber, terrainAtlasID);
	
	UVRect uv_pitcher_plant_t = addFileToAtlas("img/pitcher_plant/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_portuguese_man_of_war_t = addFileToAtlas("img/portuguese_man-of-war/top.bmp", nowSlotNumber, terrainAtlasID);

	UVRect uv_fossil_t = addFileToAtlas("img/fossil/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_fossil_s = addFileToAtlas("img/fossil/side.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_amber_t = addFileToAtlas("img/amber/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_french_bread_t = addFileToAtlas("img/french_bread/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_french_bread_s = addFileToAtlas("img/french_bread/side.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_french_bread_b = addFileToAtlas("img/french_bread/bottom.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_cambodian_school_t = addFileToAtlas("img/cambodian_school/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_cambodian_school_s = addFileToAtlas("img/cambodian_school/side.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_cambodian_school_b = addFileToAtlas("img/cambodian_school/bottom.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_eden_t = addFileToAtlas("img/eden/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_eden_s = addFileToAtlas("img/eden/side.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_x_ray_photo_t = addFileToAtlas("img/x-ray_photo/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_x_ray_photo_s = addFileToAtlas("img/x-ray_photo/side.bmp", nowSlotNumber, terrainAtlasID);
	
	UVRect uv_dice_1_t = addFileToAtlas("img/dice_1/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_dice_456_t = addFileToAtlas("img/dice_456/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_dice_456_s = addFileToAtlas("img/dice_456/side.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_dice_456_b = addFileToAtlas("img/dice_456/bottom.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_three_bamboos_t = addFileToAtlas("img/three_bamboos/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_three_bamboos_s = addFileToAtlas("img/three_bamboos/side.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_three_bamboos_b = addFileToAtlas("img/three_bamboos/bottom.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_west_t = addFileToAtlas("img/west/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_red_dragon_t = addFileToAtlas("img/red_dragon/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_limited_R_t = addFileToAtlas("img/limited_R/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_limited_R_s = addFileToAtlas("img/limited_R/side.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_limited_R_b = addFileToAtlas("img/limited_R/bottom.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_limited_S_t = addFileToAtlas("img/limited_S/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_limited_S_s = addFileToAtlas("img/limited_S/side.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_limited_S_b = addFileToAtlas("img/limited_S/bottom.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_limited_P_t = addFileToAtlas("img/limited_P/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_limited_P_s = addFileToAtlas("img/limited_P/side.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_limited_P_b = addFileToAtlas("img/limited_P/bottom.bmp", nowSlotNumber, terrainAtlasID);
	
	UVRect uv_totempole_a_t = addFileToAtlas("img/totempole_a/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_totempole_a_s = addFileToAtlas("img/totempole_a/side.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_totempole_b_s = addFileToAtlas("img/totempole_b/side.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_totempole_c_s = addFileToAtlas("img/totempole_c/side.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_dogu_t = addFileToAtlas("img/dogu/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_haniwa_t = addFileToAtlas("img/haniwa/top.bmp", nowSlotNumber, terrainAtlasID);
	
	UVRect uv_mountain_t = addFileToAtlas("img/mountain/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_mountain_faith_s = addFileToAtlas("img/mountain_faith/side.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_mountain_edge_s = addFileToAtlas("img/mountain_edge/side.bmp", nowSlotNumber, terrainAtlasID);
	
	UVRect uv_shinto_t = addFileToAtlas("img/shinto/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_buddhism_t = addFileToAtlas("img/buddhism/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_taoism_t = addFileToAtlas("img/taoism/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_hinduism_t = addFileToAtlas("img/hinduism/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_judaism_t = addFileToAtlas("img/judaism/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_christianity_t = addFileToAtlas("img/christianity/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_islam_t = addFileToAtlas("img/islam/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_atheist_t = addFileToAtlas("img/atheist/top.bmp", nowSlotNumber, terrainAtlasID);
	
	UVRect uv_outdoor_unit_t = addFileToAtlas("img/outdoor_unit/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_outdoor_unit_s = addFileToAtlas("img/outdoor_unit/side.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_gas_meter_t = addFileToAtlas("img/gas_meter/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_gas_meter_s = addFileToAtlas("img/gas_meter/side.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_garbage_bag_t = addFileToAtlas("img/garbage_bag/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_garbage_bag_s = addFileToAtlas("img/garbage_bag/side.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_designated_garbage_bag_t = addFileToAtlas("img/designated_garbage_bag/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_designated_garbage_bag_s = addFileToAtlas("img/designated_garbage_bag/side.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_designated_garbage_bag_b = addFileToAtlas("img/designated_garbage_bag/bottom.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_turntable_t = addFileToAtlas("img/turntable/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_turntable_s = addFileToAtlas("img/turntable/side.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_mirrorball_t = addFileToAtlas("img/mirrorball/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_microwave_oven_t = addFileToAtlas("img/microwave_oven/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_microwave_oven_s = addFileToAtlas("img/microwave_oven/side.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_refrigerator_t = addFileToAtlas("img/refrigerator/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_refrigerator_s = addFileToAtlas("img/refrigerator/side.bmp", nowSlotNumber, terrainAtlasID);
	
	UVRect uv_mashroom_stem_t = addFileToAtlas("img/mashroom_stem/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_poisonous_mashroom_cap_t = addFileToAtlas("img/poisonous_mashroom_cap/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_inedible_mashroom_cap_t = addFileToAtlas("img/inedible_mashroom_cap/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_inedible_mashroom_cap_b = addFileToAtlas("img/inedible_mashroom_cap/bottom.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_veiled_lady_mashroom_stem_t = addFileToAtlas("img/veiled_lady_mashroom_stem/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_veiled_lady_mashroom_cap_t = addFileToAtlas("img/veiled_lady_mashroom_cap/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_hydnellum_peckii_t = addFileToAtlas("img/hydnellum_peckii/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_octopus_fungus_t = addFileToAtlas("img/octopus_fungus/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_cordyceps_sinensis_t = addFileToAtlas("img/cordyceps_sinensis/top.bmp", nowSlotNumber, terrainAtlasID);
	
	UVRect uv_peke_t = addFileToAtlas("img/peke/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_peke_b = addFileToAtlas("img/peke/bottom.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_pixev_t = addFileToAtlas("img/pixev/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_pixev_b = addFileToAtlas("img/pixev/bottom.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_discard_t = addFileToAtlas("img/discard/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_discard_b = addFileToAtlas("img/discard/bottom.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_nikoniko_t = addFileToAtlas("img/nikoniko/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_nikoniko_b = addFileToAtlas("img/nikoniko/bottom.bmp", nowSlotNumber, terrainAtlasID);
	
	UVRect uv_reply_t = addFileToAtlas("img/reply/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_repost_t = addFileToAtlas("img/repost/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_like_t = addFileToAtlas("img/like/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_bookmark_t = addFileToAtlas("img/bookmark/top.bmp", nowSlotNumber, terrainAtlasID);
	
	UVRect uv_eyeball_dad_t = addFileToAtlas("img/eyeball_dad/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_hair_eyeball_t = addFileToAtlas("img/hair_eyeball/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_konororikondomome_t = addFileToAtlas("img/konororikondomome/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_mokumokuren_t = addFileToAtlas("img/mokumokuren/top.bmp", nowSlotNumber, terrainAtlasID);
	
	UVRect uv_japan_t = addFileToAtlas("img/japan/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_rising_sun_t = addFileToAtlas("img/rising_sun/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_kcct_t = addFileToAtlas("img/kcct/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_kcct_s = addFileToAtlas("img/kcct/side.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_denkon_t = addFileToAtlas("img/denkon/top.bmp", nowSlotNumber, terrainAtlasID);
	
	UVRect uv_allay_t = addFileToAtlas("img/allay/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_bogged_t = addFileToAtlas("img/bogged/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_creeper_t = addFileToAtlas("img/creeper/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_phantom_t = addFileToAtlas("img/phantom/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_pillager_t = addFileToAtlas("img/pillager/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_sniffer_t = addFileToAtlas("img/sniffer/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_vex_t = addFileToAtlas("img/vex/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_warden_t = addFileToAtlas("img/warden/top.bmp", nowSlotNumber, terrainAtlasID);
	
	UVRect uv_the_fool_t = addFileToAtlas("img/the_fool/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_the_fool_s = addFileToAtlas("img/the_fool/side.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_the_magician_t = addFileToAtlas("img/the_magician/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_the_high_priestess_t = addFileToAtlas("img/the_high_priestess/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_the_empress_t = addFileToAtlas("img/the_empress/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_the_emperor_t = addFileToAtlas("img/the_emperor/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_the_hierophant_t = addFileToAtlas("img/the_hierophant/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_the_lovers_t = addFileToAtlas("img/the_lovers/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_the_chariot_t = addFileToAtlas("img/the_chariot/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_strength_t = addFileToAtlas("img/strength/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_the_hermit_t = addFileToAtlas("img/the_hermit/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_wheel_of_fortune_t = addFileToAtlas("img/wheel_of_fortune/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_justice_t = addFileToAtlas("img/justice/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_the_hanged_man_t = addFileToAtlas("img/the_hanged_man/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_death_t = addFileToAtlas("img/death/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_temperance_t = addFileToAtlas("img/temperance/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_the_devil_t = addFileToAtlas("img/the_devil/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_the_tower_t = addFileToAtlas("img/the_tower/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_the_star_t = addFileToAtlas("img/the_star/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_the_moon_t = addFileToAtlas("img/the_moon/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_the_sun_t = addFileToAtlas("img/the_sun/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_judgement_t = addFileToAtlas("img/judgement/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_the_world_t = addFileToAtlas("img/the_world/top.bmp", nowSlotNumber, terrainAtlasID);
	
	UVRect uv_good_t = addFileToAtlas("img/good/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_bad_t = addFileToAtlas("img/bad/top.bmp", nowSlotNumber, terrainAtlasID);
	
	UVRect uv_diode_t = addFileToAtlas("img/diode/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_diode_s = addFileToAtlas("img/diode/side.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_npn_t = addFileToAtlas("img/npn/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_npn_s = addFileToAtlas("img/npn/side.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_pnp_t = addFileToAtlas("img/pnp/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_source_t = addFileToAtlas("img/source/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_source_s = addFileToAtlas("img/source/side.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_oscillator_t = addFileToAtlas("img/oscillator/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_oscillator_s = addFileToAtlas("img/oscillator/side.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_resistance_t = addFileToAtlas("img/resistance/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_resistance_s = addFileToAtlas("img/resistance/side.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_capacitance_t = addFileToAtlas("img/capacitance/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_capacitance_s = addFileToAtlas("img/capacitance/side.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_inductance_t = addFileToAtlas("img/inductance/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_inductance_s = addFileToAtlas("img/inductance/side.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_switch_t = addFileToAtlas("img/switch/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_blackbox_t = addFileToAtlas("img/blackbox/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_blackbox_s = addFileToAtlas("img/blackbox/side.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_grand_t = addFileToAtlas("img/grand/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_grand_s = addFileToAtlas("img/grand/side.bmp", nowSlotNumber, terrainAtlasID);
	
	UVRect uv_wire_terminal_t = addFileToAtlas("img/wire_terminal/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_wire_h_t = addFileToAtlas("img/wire_h/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_wire_h_s = addFileToAtlas("img/wire_h/side.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_wire_v_t = addFileToAtlas("img/wire_v/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_wire_cross_t = addFileToAtlas("img/wire_cross/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_wire_cross_dot_t = addFileToAtlas("img/wire_cross_dot/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_wire_t_u_t = addFileToAtlas("img/wire_t_u/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_wire_t_d_t = addFileToAtlas("img/wire_t_d/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_wire_t_l_t = addFileToAtlas("img/wire_t_l/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_wire_t_r_t = addFileToAtlas("img/wire_t_r/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_wire_lu_t = addFileToAtlas("img/wire_lu/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_wire_ld_t = addFileToAtlas("img/wire_ld/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_wire_ru_t = addFileToAtlas("img/wire_ru/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_wire_rd_t = addFileToAtlas("img/wire_rd/top.bmp", nowSlotNumber, terrainAtlasID);
	
	UVRect uv_nl_0_t = addFileToAtlas("img/nl_0/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_nl_1_t = addFileToAtlas("img/nl_1/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_nl_2_t = addFileToAtlas("img/nl_2/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_nl_3_t = addFileToAtlas("img/nl_3/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_nl_4_t = addFileToAtlas("img/nl_4/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_nl_5_t = addFileToAtlas("img/nl_5/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_nl_6_t = addFileToAtlas("img/nl_6/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_nl_7_t = addFileToAtlas("img/nl_7/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_nl_8_t = addFileToAtlas("img/nl_8/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_nl_9_t = addFileToAtlas("img/nl_9/top.bmp", nowSlotNumber, terrainAtlasID);
	
	UVRect uv_al_a_t = addFileToAtlas("img/al_a/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_al_b_t = addFileToAtlas("img/al_b/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_al_c_t = addFileToAtlas("img/al_c/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_al_d_t = addFileToAtlas("img/al_d/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_al_e_t = addFileToAtlas("img/al_e/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_al_f_t = addFileToAtlas("img/al_f/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_al_g_t = addFileToAtlas("img/al_g/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_al_h_t = addFileToAtlas("img/al_h/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_al_i_t = addFileToAtlas("img/al_i/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_al_j_t = addFileToAtlas("img/al_j/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_al_k_t = addFileToAtlas("img/al_k/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_al_l_t = addFileToAtlas("img/al_l/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_al_m_t = addFileToAtlas("img/al_m/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_al_n_t = addFileToAtlas("img/al_n/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_al_o_t = addFileToAtlas("img/al_o/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_al_p_t = addFileToAtlas("img/al_p/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_al_q_t = addFileToAtlas("img/al_q/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_al_r_t = addFileToAtlas("img/al_r/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_al_s_t = addFileToAtlas("img/al_s/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_al_t_t = addFileToAtlas("img/al_t/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_al_u_t = addFileToAtlas("img/al_u/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_al_v_t = addFileToAtlas("img/al_v/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_al_w_t = addFileToAtlas("img/al_w/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_al_x_t = addFileToAtlas("img/al_x/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_al_y_t = addFileToAtlas("img/al_y/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_al_z_t = addFileToAtlas("img/al_z/top.bmp", nowSlotNumber, terrainAtlasID);
	
	UVRect uv_sl_space_t = addFileToAtlas("img/sl_space/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_sl_period_t = addFileToAtlas("img/sl_period/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_sl_comma_t = addFileToAtlas("img/sl_comma/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_sl_question_t = addFileToAtlas("img/sl_question/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_sl_exclamation_t = addFileToAtlas("img/sl_exclamation/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_sl_plus_t = addFileToAtlas("img/sl_plus/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_sl_minus_t = addFileToAtlas("img/sl_minus/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_sl_cross_t = addFileToAtlas("img/sl_cross/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_sl_slash_t = addFileToAtlas("img/sl_slash/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_sl_equal_t = addFileToAtlas("img/sl_equal/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_sl_parentesis_open_t = addFileToAtlas("img/sl_parentesis_open/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_sl_parentesis_close_t = addFileToAtlas("img/sl_parentesis_close/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_sl_colon_t = addFileToAtlas("img/sl_colon/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_sl_semicolon_t = addFileToAtlas("img/sl_semicolon/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_sl_single_quatation_t = addFileToAtlas("img/sl_single_quotation/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_sl_double_quatation_t = addFileToAtlas("img/sl_double_quotation/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_sl_at_t = addFileToAtlas("img/sl_at/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_sl_tilde_t = addFileToAtlas("img/sl_tilde/top.bmp", nowSlotNumber, terrainAtlasID);	
	UVRect uv_sl_hat_t = addFileToAtlas("img/sl_hat/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_sl_underbar_t = addFileToAtlas("img/sl_underbar/top.bmp", nowSlotNumber, terrainAtlasID);
	
	UVRect uv_cl_higashi_t = addFileToAtlas("img/cl_higashi/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_cl_minami_t = addFileToAtlas("img/cl_minami/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_cl_nishi_t = addFileToAtlas("img/cl_nishi/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_cl_kita_t = addFileToAtlas("img/cl_kita/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_cl_naka_t = addFileToAtlas("img/cl_naka/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_cl_yama_t = addFileToAtlas("img/cl_yama/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_cl_ishi_t = addFileToAtlas("img/cl_ishi/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_cl_ki_t = addFileToAtlas("img/cl_ki/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_cl_michi_a_t = addFileToAtlas("img/cl_michi_a/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_cl_michi_b_t = addFileToAtlas("img/cl_michi_b/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_cl_michi_c_t = addFileToAtlas("img/cl_michi_c/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_cl_michi_d_t = addFileToAtlas("img/cl_michi_d/top.bmp", nowSlotNumber, terrainAtlasID);
	
	UVRect uv_cl_kokoro_t = addFileToAtlas("img/cl_kokoro/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_cl_waza_t = addFileToAtlas("img/cl_waza/top.bmp", nowSlotNumber, terrainAtlasID);
	UVRect uv_cl_karada_t = addFileToAtlas("img/cl_karada/top.bmp", nowSlotNumber, terrainAtlasID);
	
	blocks.emplace_back("Air", Skin(0), false, false, false);
	
	blocks.emplace_back("Soil", Skin(0, uv_soil_t));
	blocks.emplace_back("Grass", Skin(2, uv_grass_t, uv_grass_s, uv_soil_t));
	
	blocks.emplace_back("Stone", Skin(0, uv_stone_t));
	blocks.emplace_back("Cobblestone", Skin(0, uv_cobblestone_t));
	blocks.emplace_back("Deepslate", Skin(0, uv_deepslate_t));
	blocks.emplace_back("Obsidian", Skin(0, uv_obsidian_t));

	blocks.emplace_back("Granite", Skin(0, uv_granite_t));
	blocks.emplace_back("Diorite", Skin(0, uv_diorite_t));
	blocks.emplace_back("Gabbro", Skin(0, uv_gabbro_t));
	blocks.emplace_back("Rhyolite", Skin(0, uv_rhyolite_t));
	blocks.emplace_back("Basalt", Skin(1, uv_basalt_t, uv_basalt_s));
	blocks.emplace_back("Andesite", Skin(0, uv_andesite_t));
	
	blocks.emplace_back("Sand", Skin(0, uv_sand_t));
	blocks.emplace_back("Sandstone", Skin(0, uv_sandstone_t));
	blocks.emplace_back("Mud", Skin(0, uv_mud_t));
	blocks.emplace_back("Mudstone", Skin(0, uv_mudstone_t));
	
	blocks.emplace_back("Iron", Skin(0, uv_iron_t));
	blocks.emplace_back("Copper", Skin(0, uv_copper_t));
	blocks.emplace_back("Gold", Skin(0, uv_gold_t));
	blocks.emplace_back("Tin", Skin(0, uv_tin_t));
	blocks.emplace_back("Platinum", Skin(0, uv_platinum_t));
	blocks.emplace_back("Uranium", Skin(0, uv_uranium_t));
	
	blocks.emplace_back("Water", Skin(0, uv_water_t), false, true, false);
	blocks.emplace_back("Lava", Skin(0, uv_lava_t), false, true, false);
	
	blocks.emplace_back("Brick", Skin(2, uv_brick_t, uv_brick_s, uv_brick_b));
	
	blocks.emplace_back("Oak Log", Skin(1, uv_oak_log_t, uv_oak_log_s));
	blocks.emplace_back("Spruce Log", Skin(1, uv_spruce_log_t, uv_spruce_log_s));
	blocks.emplace_back("Birch Log", Skin(1, uv_birch_log_t, uv_birch_log_s));
	
	blocks.emplace_back("Leaf", Skin(0, uv_leaf_t));
	
	blocks.emplace_back("Oak Wood", Skin(0, uv_oak_wood_t));
	blocks.emplace_back("Spruce Wood", Skin(0, uv_spruce_wood_t));
	blocks.emplace_back("Birch Wood", Skin(0, uv_birch_wood_t));
	
	blocks.emplace_back("White Wool", Skin(0, uv_wool_white_t));
	blocks.emplace_back("Light Gray Wool", Skin(0, uv_wool_lightgray_t));
	blocks.emplace_back("Gray Wool", Skin(0, uv_wool_gray_t));
	blocks.emplace_back("Black Wool", Skin(0, uv_wool_black_t));
	blocks.emplace_back("Brown Wool", Skin(0, uv_wool_brown_t));
	blocks.emplace_back("Red Wool", Skin(0, uv_wool_red_t));
	blocks.emplace_back("Orange Wool", Skin(0, uv_wool_orange_t));
	blocks.emplace_back("Yellow Wool", Skin(0, uv_wool_yellow_t));
	blocks.emplace_back("Lime Wool", Skin(0, uv_wool_lime_t));
	blocks.emplace_back("Green Wool", Skin(0, uv_wool_green_t));
	blocks.emplace_back("Cyan Wool", Skin(0, uv_wool_cyan_t));
	blocks.emplace_back("Sky Blue Wool", Skin(0, uv_wool_skyblue_t));
	blocks.emplace_back("Blue Wool", Skin(0, uv_wool_blue_t));
	blocks.emplace_back("Magenta Wool", Skin(0, uv_wool_magenta_t));
	blocks.emplace_back("Purple Wool", Skin(0, uv_wool_purple_t));
	blocks.emplace_back("Pink Wool", Skin(0, uv_wool_pink_t));
	
	blocks.emplace_back("White Concrete", Skin(0, uv_concrete_white_t));
	blocks.emplace_back("Light Gray Concrete", Skin(0, uv_concrete_lightgray_t));
	blocks.emplace_back("Gray Concrete", Skin(0, uv_concrete_gray_t));
	blocks.emplace_back("Black Concrete", Skin(0, uv_concrete_black_t));
	blocks.emplace_back("Brown Concrete", Skin(0, uv_concrete_brown_t));
	blocks.emplace_back("Red Concrete", Skin(0, uv_concrete_red_t));
	blocks.emplace_back("Orange Concrete", Skin(0, uv_concrete_orange_t));
	blocks.emplace_back("Yellow Concrete", Skin(0, uv_concrete_yellow_t));
	blocks.emplace_back("Lime Concrete", Skin(0, uv_concrete_lime_t));
	blocks.emplace_back("Green Concrete", Skin(0, uv_concrete_green_t));
	blocks.emplace_back("Cyan Concrete", Skin(0, uv_concrete_cyan_t));
	blocks.emplace_back("Sky Blue Concrete", Skin(0, uv_concrete_skyblue_t));
	blocks.emplace_back("Blue Concrete", Skin(0, uv_concrete_blue_t));
	blocks.emplace_back("Magenta Concrete", Skin(0, uv_concrete_magenta_t));
	blocks.emplace_back("Purple Concrete", Skin(0, uv_concrete_purple_t));
	blocks.emplace_back("Pink Concrete", Skin(0, uv_concrete_pink_t));
	
	blocks.emplace_back("Pitcher Plant", Skin(0, uv_pitcher_plant_t));
	blocks.emplace_back("Portuguese Man-of-War", Skin(0, uv_portuguese_man_of_war_t));
	blocks.emplace_back("Amber", Skin(0, uv_amber_t));
	blocks.emplace_back("Fossil", Skin(1, uv_fossil_t, uv_fossil_s));
	blocks.emplace_back("French Bread", Skin(2, uv_french_bread_t, uv_french_bread_s, uv_french_bread_b));
	blocks.emplace_back("Cambodian School", Skin(2, uv_cambodian_school_t, uv_cambodian_school_s, uv_cambodian_school_b));
	blocks.emplace_back("Eden", Skin(1, uv_eden_t, uv_eden_s));
	blocks.emplace_back("X-Ray Photo", Skin(1, uv_x_ray_photo_t, uv_x_ray_photo_s));
	
	blocks.emplace_back("Dice but all 1", Skin(1, uv_dice_1_t, uv_dice_1_t));
	blocks.emplace_back("Dice but only 456", Skin(2, uv_dice_456_t, uv_dice_456_s, uv_dice_456_b));
	blocks.emplace_back("Three Bamboos", Skin(2, uv_three_bamboos_t, uv_three_bamboos_s, uv_three_bamboos_b));
	blocks.emplace_back("West", Skin(2, uv_west_t, uv_three_bamboos_s, uv_three_bamboos_b));
	blocks.emplace_back("Red Dragon", Skin(2, uv_red_dragon_t, uv_three_bamboos_s, uv_three_bamboos_b));
	blocks.emplace_back("Limited JangKeng Rock", Skin(2, uv_limited_R_t, uv_limited_R_s, uv_limited_R_b));
	blocks.emplace_back("Limited JangKeng Scissors", Skin(2, uv_limited_S_t, uv_limited_S_s, uv_limited_S_b));
	blocks.emplace_back("Limited JangKeng Paper", Skin(2, uv_limited_P_t, uv_limited_P_s, uv_limited_P_b));
	
	blocks.emplace_back("Totem Pole A", Skin(1, uv_totempole_a_t, uv_totempole_a_s));
	blocks.emplace_back("Totem Pole B", Skin(1, uv_totempole_a_t, uv_totempole_b_s));
	blocks.emplace_back("Totem Pole C", Skin(1, uv_totempole_a_t, uv_totempole_c_s));
	blocks.emplace_back("Dogu", Skin(0, uv_dogu_t));
	blocks.emplace_back("Haniwa", Skin(0, uv_haniwa_t));
	
	blocks.emplace_back("Mountain", Skin(0, uv_mountain_t));
	blocks.emplace_back("Mountain Faith", Skin(1, uv_mountain_t, uv_mountain_faith_s));
	blocks.emplace_back("Mountain Edge", Skin(1, uv_mountain_t, uv_mountain_edge_s));
	
	blocks.emplace_back("Shinto", Skin(0, uv_shinto_t));
	blocks.emplace_back("Buddhism", Skin(0, uv_buddhism_t));
	blocks.emplace_back("Taoism", Skin(0, uv_taoism_t));
	blocks.emplace_back("Hinduism", Skin(0, uv_hinduism_t));
	blocks.emplace_back("Judaism", Skin(0, uv_judaism_t));
	blocks.emplace_back("Christianity", Skin(0, uv_christianity_t));
	blocks.emplace_back("Islam", Skin(0, uv_islam_t));
	blocks.emplace_back("Atheist", Skin(0, uv_atheist_t));
	
	blocks.emplace_back("Outdoor Unit", Skin(1, uv_outdoor_unit_t, uv_outdoor_unit_s));
	blocks.emplace_back("Gas Meter", Skin(1, uv_gas_meter_t, uv_gas_meter_s));
	blocks.emplace_back("Garbage Bag", Skin(2, uv_garbage_bag_t, uv_garbage_bag_s, uv_garbage_bag_s));
	blocks.emplace_back("Designated Garbage Bag", Skin(3, uv_designated_garbage_bag_t, uv_designated_garbage_bag_s, uv_designated_garbage_bag_b));
	blocks.emplace_back("Turntable", Skin(1, uv_turntable_t, uv_turntable_s));
	blocks.emplace_back("Mirrorball", Skin(0, uv_mirrorball_t));
	blocks.emplace_back("Microwave Oven", Skin(1, uv_microwave_oven_t, uv_microwave_oven_s));
	blocks.emplace_back("Refrigerator", Skin(1, uv_refrigerator_t, uv_refrigerator_s));
	
	blocks.emplace_back("Mashroom Stem", Skin(0, uv_mashroom_stem_t));
	blocks.emplace_back("Poisonous Mashroom Cap", Skin(0, uv_poisonous_mashroom_cap_t));
	blocks.emplace_back("Inedible Mashroom Cap", Skin(2, uv_inedible_mashroom_cap_t, uv_inedible_mashroom_cap_t, uv_inedible_mashroom_cap_b));
	blocks.emplace_back("Veiled Lady Mashroom Stem", Skin(0, uv_veiled_lady_mashroom_stem_t));
	blocks.emplace_back("Veiled Lady Mashroom Cap", Skin(0, uv_veiled_lady_mashroom_cap_t));
	blocks.emplace_back("Hydnellum Peckii", Skin(0, uv_hydnellum_peckii_t));
	blocks.emplace_back("Octopus Fungus", Skin(0, uv_octopus_fungus_t));
	blocks.emplace_back("Cordyceps Sinensis", Skin(0, uv_cordyceps_sinensis_t));
	
	blocks.emplace_back("Peke", Skin(2, uv_peke_t, uv_peke_t, uv_peke_b));
	blocks.emplace_back("Pixev", Skin(2, uv_pixev_t, uv_pixev_t, uv_pixev_b));
	blocks.emplace_back("Discard", Skin(2, uv_discard_t, uv_discard_t, uv_discard_b));
	blocks.emplace_back("Nikoniko", Skin(2, uv_nikoniko_t, uv_nikoniko_t, uv_nikoniko_b));
	
	blocks.emplace_back("Reply", Skin(0, uv_reply_t));
	blocks.emplace_back("Repost", Skin(0, uv_repost_t));
	blocks.emplace_back("Like", Skin(0, uv_like_t));
	blocks.emplace_back("Bookmark", Skin(0, uv_bookmark_t));
	
	blocks.emplace_back("Eyeball Dad", Skin(0, uv_eyeball_dad_t));
	blocks.emplace_back("Hair Eyeball", Skin(0, uv_hair_eyeball_t));
	blocks.emplace_back("Konororikondomome", Skin(0, uv_konororikondomome_t));
	blocks.emplace_back("Mokumokuren", Skin(0, uv_mokumokuren_t));
	
	blocks.emplace_back("Japan", Skin(0, uv_japan_t));
	blocks.emplace_back("Rising Sun", Skin(0, uv_rising_sun_t));
	blocks.emplace_back("KCCT", Skin(1, uv_kcct_t, uv_kcct_s));
	blocks.emplace_back("Denkon", Skin(0, uv_denkon_t));
	
	blocks.emplace_back("Allay", Skin(0, uv_allay_t));
	blocks.emplace_back("Bogged", Skin(0, uv_bogged_t));
	blocks.emplace_back("Creeper", Skin(0, uv_creeper_t));
	blocks.emplace_back("Phantom", Skin(0, uv_phantom_t));
	blocks.emplace_back("Pillager", Skin(0, uv_pillager_t));
	blocks.emplace_back("Sniffer", Skin(0, uv_sniffer_t));
	blocks.emplace_back("Vex", Skin(0, uv_vex_t));
	blocks.emplace_back("Warden", Skin(0, uv_warden_t));
	
	blocks.emplace_back("Tarot: The Fool", Skin(2, uv_the_fool_t, uv_the_fool_s, uv_the_fool_s));
	blocks.emplace_back("Tarot: The Magician", Skin(2, uv_the_magician_t, uv_the_fool_s, uv_the_fool_s));
	blocks.emplace_back("Tarot: The High Priestess", Skin(2, uv_the_high_priestess_t, uv_the_fool_s, uv_the_fool_s));
	blocks.emplace_back("Tarot: The Empress", Skin(2, uv_the_empress_t, uv_the_fool_s, uv_the_fool_s));
	blocks.emplace_back("Tarot: The Emperor", Skin(2, uv_the_emperor_t, uv_the_fool_s, uv_the_fool_s));
	blocks.emplace_back("Tarot: The Hierophant", Skin(2, uv_the_hierophant_t, uv_the_fool_s, uv_the_fool_s));
	blocks.emplace_back("Tarot: The Lovers", Skin(2, uv_the_lovers_t, uv_the_fool_s, uv_the_fool_s));
	blocks.emplace_back("Tarot: The Chariot", Skin(2, uv_the_chariot_t, uv_the_fool_s, uv_the_fool_s));
	blocks.emplace_back("Tarot: Strength", Skin(2, uv_strength_t, uv_the_fool_s, uv_the_fool_s));
	blocks.emplace_back("Tarot: The Hermit", Skin(2, uv_the_hermit_t, uv_the_fool_s, uv_the_fool_s));
	blocks.emplace_back("Tarot: Wheel of Fortune", Skin(2, uv_wheel_of_fortune_t, uv_the_fool_s, uv_the_fool_s));
	blocks.emplace_back("Tarot: Justice", Skin(2, uv_justice_t, uv_the_fool_s, uv_the_fool_s));
	blocks.emplace_back("Tarot: The Hanged Man", Skin(2, uv_the_hanged_man_t, uv_the_fool_s, uv_the_fool_s));
	blocks.emplace_back("Tarot: Death", Skin(2, uv_death_t, uv_the_fool_s, uv_the_fool_s));
	blocks.emplace_back("Tarot: Temperance", Skin(2, uv_temperance_t, uv_the_fool_s, uv_the_fool_s));
	blocks.emplace_back("Tarot: The Devil", Skin(2, uv_the_devil_t, uv_the_fool_s, uv_the_fool_s));
	blocks.emplace_back("Tarot: The Tower", Skin(2, uv_the_tower_t, uv_the_fool_s, uv_the_fool_s));
	blocks.emplace_back("Tarot: The Star", Skin(2, uv_the_star_t, uv_the_fool_s, uv_the_fool_s));
	blocks.emplace_back("Tarot: The Moon", Skin(2, uv_the_moon_t, uv_the_fool_s, uv_the_fool_s));
	blocks.emplace_back("Tarot: The Sun", Skin(2, uv_the_sun_t, uv_the_fool_s, uv_the_fool_s));
	blocks.emplace_back("Tarot: Judgement", Skin(2, uv_judgement_t, uv_the_fool_s, uv_the_fool_s));
	blocks.emplace_back("Tarot: The World", Skin(2, uv_the_world_t, uv_the_fool_s, uv_the_fool_s));
	
	blocks.emplace_back("Good", Skin(1, uv_good_t, uv_good_t));
	blocks.emplace_back("Bad", Skin(1, uv_bad_t, uv_bad_t));
	
	blocks.emplace_back("Diode", Skin(1, uv_diode_t, uv_diode_s));
	blocks.emplace_back("NPN Transistor", Skin(1, uv_npn_t, uv_npn_s));
	blocks.emplace_back("PNP Transistor", Skin(1, uv_pnp_t, uv_npn_s));
	blocks.emplace_back("Source", Skin(1, uv_source_t, uv_source_s));
	blocks.emplace_back("Oscillator", Skin(1, uv_oscillator_t, uv_oscillator_s));
	blocks.emplace_back("Resistance", Skin(1, uv_resistance_t, uv_resistance_s));
	blocks.emplace_back("Capacitance", Skin(1, uv_capacitance_t, uv_capacitance_s));
	blocks.emplace_back("Inductance", Skin(1, uv_inductance_t, uv_inductance_s));
	blocks.emplace_back("Switch", Skin(1, uv_switch_t, uv_wire_h_s));
	blocks.emplace_back("Blackbox", Skin(1, uv_blackbox_t, uv_blackbox_s));
	blocks.emplace_back("Grand", Skin(1, uv_grand_t, uv_grand_s));
	
	blocks.emplace_back("Terminal", Skin(1, uv_wire_terminal_t, uv_wire_h_s));
	blocks.emplace_back("Wire Horizontal", Skin(1, uv_wire_h_t, uv_wire_h_s));
	blocks.emplace_back("Wire Vertical", Skin(1, uv_wire_v_t, uv_wire_h_s));
	blocks.emplace_back("Wire Cross", Skin(1, uv_wire_cross_t, uv_wire_h_s));
	blocks.emplace_back("Wire Cross with Dot", Skin(1, uv_wire_cross_dot_t, uv_wire_h_s));
	blocks.emplace_back("Wire T Up", Skin(1, uv_wire_t_u_t, uv_wire_h_s));
	blocks.emplace_back("Wire T Down", Skin(1, uv_wire_t_d_t, uv_wire_h_s));
	blocks.emplace_back("Wire T Left", Skin(1, uv_wire_t_l_t, uv_wire_h_s));
	blocks.emplace_back("Wire T Right", Skin(1, uv_wire_t_r_t, uv_wire_h_s));
	blocks.emplace_back("Wire Corner Left Up", Skin(1, uv_wire_lu_t, uv_wire_h_s));
	blocks.emplace_back("Wire Corner Left Down", Skin(1, uv_wire_ld_t, uv_wire_h_s));
	blocks.emplace_back("Wire Corner Right Up", Skin(1, uv_wire_ru_t, uv_wire_h_s));
	blocks.emplace_back("Wire Corner Right Down", Skin(1, uv_wire_rd_t, uv_wire_h_s));
	
	blocks.emplace_back("Number '0'", Skin(0, uv_nl_0_t));
	blocks.emplace_back("Number '1'", Skin(0, uv_nl_1_t));
	blocks.emplace_back("Number '2'", Skin(0, uv_nl_2_t));
	blocks.emplace_back("Number '3'", Skin(0, uv_nl_3_t));
	blocks.emplace_back("Number '4'", Skin(0, uv_nl_4_t));
	blocks.emplace_back("Number '5'", Skin(0, uv_nl_5_t));
	blocks.emplace_back("Number '6'", Skin(0, uv_nl_6_t));
	blocks.emplace_back("Number '7'", Skin(0, uv_nl_7_t));
	blocks.emplace_back("Number '8'", Skin(0, uv_nl_8_t));
	blocks.emplace_back("Number '9'", Skin(0, uv_nl_9_t));
	
	blocks.emplace_back("Alphabet 'A'", Skin(0, uv_al_a_t));
	blocks.emplace_back("Alphabet 'B'", Skin(0, uv_al_b_t));
	blocks.emplace_back("Alphabet 'C'", Skin(0, uv_al_c_t));
	blocks.emplace_back("Alphabet 'D'", Skin(0, uv_al_d_t));
	blocks.emplace_back("Alphabet 'E'", Skin(0, uv_al_e_t));
	blocks.emplace_back("Alphabet 'F'", Skin(0, uv_al_f_t));
	blocks.emplace_back("Alphabet 'G'", Skin(0, uv_al_g_t));
	blocks.emplace_back("Alphabet 'H'", Skin(0, uv_al_h_t));
	blocks.emplace_back("Alphabet 'I'", Skin(0, uv_al_i_t));
	blocks.emplace_back("Alphabet 'J'", Skin(0, uv_al_j_t));
	blocks.emplace_back("Alphabet 'K'", Skin(0, uv_al_k_t));
	blocks.emplace_back("Alphabet 'L'", Skin(0, uv_al_l_t));
	blocks.emplace_back("Alphabet 'M'", Skin(0, uv_al_m_t));
	blocks.emplace_back("Alphabet 'N'", Skin(0, uv_al_n_t));
	blocks.emplace_back("Alphabet 'O'", Skin(0, uv_al_o_t));
	blocks.emplace_back("Alphabet 'P'", Skin(0, uv_al_p_t));
	blocks.emplace_back("Alphabet 'Q'", Skin(0, uv_al_q_t));
	blocks.emplace_back("Alphabet 'R'", Skin(0, uv_al_r_t));
	blocks.emplace_back("Alphabet 'S'", Skin(0, uv_al_s_t));
	blocks.emplace_back("Alphabet 'T'", Skin(0, uv_al_t_t));
	blocks.emplace_back("Alphabet 'U'", Skin(0, uv_al_u_t));
	blocks.emplace_back("Alphabet 'V'", Skin(0, uv_al_v_t));
	blocks.emplace_back("Alphabet 'W'", Skin(0, uv_al_w_t));
	blocks.emplace_back("Alphabet 'X'", Skin(0, uv_al_x_t));
	blocks.emplace_back("Alphabet 'Y'", Skin(0, uv_al_y_t));
	blocks.emplace_back("Alphabet 'Z'", Skin(0, uv_al_z_t));
	
	blocks.emplace_back("Symbol Space", Skin(0, uv_sl_space_t));
	blocks.emplace_back("Symbol Period", Skin(0, uv_sl_period_t));
	blocks.emplace_back("Symbol Comma", Skin(0, uv_sl_comma_t));
	blocks.emplace_back("Symbol Question", Skin(0, uv_sl_question_t));
	blocks.emplace_back("Symbol Exclamation", Skin(0, uv_sl_exclamation_t));
	blocks.emplace_back("Symbol Plus", Skin(0, uv_sl_plus_t));
	blocks.emplace_back("Symbol Minus", Skin(0, uv_sl_minus_t));
	blocks.emplace_back("Symbol Cross", Skin(0, uv_sl_cross_t));
	blocks.emplace_back("Symbol Slash", Skin(0, uv_sl_slash_t));
	blocks.emplace_back("Symbol Equal", Skin(0, uv_sl_equal_t));
	blocks.emplace_back("Symbol Parentesis Open", Skin(0, uv_sl_parentesis_open_t));
	blocks.emplace_back("Symbol Parentesis Close", Skin(0, uv_sl_parentesis_close_t));
	blocks.emplace_back("Symbol Colon", Skin(0, uv_sl_colon_t));
	blocks.emplace_back("Symbol Semicolon", Skin(0, uv_sl_semicolon_t));
	blocks.emplace_back("Symbol Single Quatation", Skin(0, uv_sl_single_quatation_t));
	blocks.emplace_back("Symbol Double Quatation", Skin(0, uv_sl_double_quatation_t));
	blocks.emplace_back("Symbol At", Skin(0, uv_sl_at_t));
	blocks.emplace_back("Symbol Tilde", Skin(0, uv_sl_tilde_t));
	blocks.emplace_back("Symbol Hat", Skin(0, uv_sl_hat_t));
	blocks.emplace_back("Symbol Underbar", Skin(0, uv_sl_underbar_t));
	
	blocks.emplace_back("Kanji Higashi", Skin(0, uv_cl_higashi_t));
	blocks.emplace_back("Kanji Minami", Skin(0, uv_cl_minami_t));
	blocks.emplace_back("Kanji Nishi", Skin(0, uv_cl_nishi_t));
	blocks.emplace_back("Kanji Kita", Skin(0, uv_cl_kita_t));
	blocks.emplace_back("Kanji Naka", Skin(0, uv_cl_naka_t));
	blocks.emplace_back("Kanji Yama", Skin(0, uv_cl_yama_t));
	blocks.emplace_back("Kanji Ishi", Skin(0, uv_cl_ishi_t));
	blocks.emplace_back("Kanji Ki", Skin(0, uv_cl_ki_t));
	blocks.emplace_back("Kanji Michi A", Skin(0, uv_cl_michi_a_t));
	blocks.emplace_back("Kanji Michi B", Skin(0, uv_cl_michi_b_t));
	blocks.emplace_back("Kanji Michi C", Skin(0, uv_cl_michi_c_t));
	blocks.emplace_back("Kanji Michi D", Skin(0, uv_cl_michi_d_t));
	blocks.emplace_back("Kanji Kokoro", Skin(0, uv_cl_kokoro_t));
	blocks.emplace_back("Kanji Waza", Skin(0, uv_cl_waza_t));
	blocks.emplace_back("Kanji Karada", Skin(0, uv_cl_karada_t));
}

Block* BlockRegistry::get(int id) {
	if(id >= 0 && id < (int)blocks.size()) {
		return &blocks[id];
	}
	return &blocks[0];
}
