#include "common.h"

Box::Box(Vector min = Vector(), Vector max = Vector()) : min(min), max(max) {}
	
Box Box::move(float x, float y, float z) {
	return Box(Vector(min.x + x, min.y + y, min.z + z), Vector(max.x + x, max.y + y, max.z + z));
}
	
Vector Box::center() {
	return Vector((min.x + max.x) / 2, (min.y + max.y) / 2, (min.z + max.z) / 2);
}
	
bool Box::intersects(const Box& other) {
	return (min.x < other.max.x && max.x > other.min.x) && (min.y < other.max.y && max.y > other.min.y) && (min.z < other.max.z && max.z > other.min.z);
}


Player::Player() {
	pos = Vector(64.0f, 16.0f, 64.0f);
	vel = Vector(0.0f, 0.0f, 0.0f);
	insideVel = Vector(0.0f, 0.0f, 0.0f);
	outsideVel = Vector(0.0f, 0.0f, 0.0f);
	camera = Angle(0.0f, 0.0f);
	moveSpeed = 0.075f;
	isFlying = false;
	isGrounded = false;
	hitbox = Box(Vector(-0.3f, 0.0f, -0.3f), Vector(0.3f, 1.8f, 0.3f));
	handBlockId = 1;
}

void Player::walk(Keyboard &keyboard) {
	insideVel.x = 0.0f;
	insideVel.y = 0.0f;
	insideVel.z = 0.0f;
	
	bool isWalking = false;
	float walkDir = camera.azi;
	if(keyboard.getKey('w').pressed) {
		isWalking = true;
		if(keyboard.getKey('a').pressed) {
			walkDir += -PI / 4;
		} else if(keyboard.getKey('d').pressed) {
			walkDir += PI / 4;
		} else {
			walkDir += 0.0f;
		}
	} else if(keyboard.getKey('s').pressed) {
		isWalking = true;
		if(keyboard.getKey('a').pressed) {
			walkDir += -3 * PI / 4;
		} else if(keyboard.getKey('d').pressed) {
			walkDir += 3 * PI / 4;
		} else {
			walkDir += PI;
		}
	}else if(keyboard.getKey('a').pressed) {
		isWalking = true;
		walkDir += -PI / 2;
	}else if(keyboard.getKey('d').pressed) {
		isWalking = true;
		walkDir += PI / 2;
	}
	if(isWalking){
		insideVel.x = moveSpeed * sin(walkDir) * (isFlying ? 2.0f : 1.0f);
		insideVel.z = -moveSpeed * cos(walkDir) * (isFlying ? 2.0f : 1.0f);
	}
	
	
	if(keyboard.getKey(32).pressed) {
		if(isGrounded && !isFlying) {
			outsideVel.y = moveSpeed * 2;
			isGrounded = false;
		}
		if(isFlying) {
			insideVel.y += moveSpeed * 2.0;
		}
	}

	if(keyboard.getKey(112 + 256).pressed) {
		if(isFlying) {
			insideVel.y -= moveSpeed * 2.0;
		}
	}
}

bool Player::checkCollision(Vector p, World &world) {
	Box playerBox = hitbox.move(p.x, p.y, p.z);

	for (int x = (int)floor(playerBox.min.x); x <= (int)floor(playerBox.max.x); x++) {
		for (int y = (int)floor(playerBox.min.y); y <= (int)floor(playerBox.max.y); y++) {
			for (int z = (int)floor(playerBox.min.z); z <= (int)floor(playerBox.max.z); z++) {
				if (x >= 0 && x < world.width && y >= 0 && y < world.height && z >= 0 && z < world.depth) {
					if (world.blockRegistry.get(world.getBlock(x, y, z))->isCollidable) {
						Box blockBox(Vector(x, y, z), Vector(x + 1, y + 1, z + 1));
						if (playerBox.intersects(blockBox)) {
							return true;
						}
					}
				}
			}
		}
	}
	return false;
}
	
void Player::update(World &world) {
	if(isFlying) {
		outsideVel.y = 0;
	} else {
		outsideVel.y -= 0.005f;
	}
	if (outsideVel.y < -0.5f) {
		outsideVel.y = -0.5f;
	}
	vel = insideVel + outsideVel;
	
	pos.y += vel.y;
	if (checkCollision(pos, world)) {
		isGrounded = true;
		outsideVel.y = 0.0;
		pos.y -= vel.y;
	} else {
		isGrounded = false;
	}
	
	if(pos.y <= 0.0){
		pos.y = 0.0;
		outsideVel.y = 0.0;
		isGrounded = true;
	}
	
	pos.x += vel.x;
	if (checkCollision(pos, world)) {
		outsideVel.x = 0.0;
		pos.x -= vel.x;
	}
	pos.z += vel.z;
	if (checkCollision(pos, world)) {
		outsideVel.z = 0.0;
		pos.z -= vel.z;
	}
}
