#include "common.h"

Chunk::Chunk() : listId(0), needsUpdate(true) {
	memset(blocks, 0, sizeof(blocks));
}

Chunk::~Chunk() {
	if(listId != 0) {
		glDeleteLists(listId, 1);
	}
}

void Chunk::init(int x, int y, int z) {
	cx = x; cy = y; cz = z;
}

bool Chunk::shouldRenderFace(World& world, int gx, int gy, int gz) {
	int id = world.getBlock(gx, gy, gz);
	Block* b = world.blockRegistry.get(id);
	return b ? !b->isOpaque : true;
}

void Chunk::updateDisplayList(World& world, GLuint terrainAtlasID) {
	if(listId == 0){
		listId = glGenLists(1);
	}
	glNewList(listId, GL_COMPILE);
	
	glEnable(GL_TEXTURE_2D);
	glBindTexture(GL_TEXTURE_2D, terrainAtlasID);
	glColor3f(1.0f, 1.0f, 1.0f);

	glBegin(GL_QUADS);

	for(int y = 0; y < CHUNK_SIZE; y ++) {
		for(int x = 0; x < CHUNK_SIZE; x ++) {
			for(int z = 0; z < CHUNK_SIZE; z ++) {
				int id = blocks[x][y][z];
				Block* block = world.blockRegistry.get(id);
				
				if(!block || !block->isDisplayed) {
					continue;
				}
				
				int gx = cx * CHUNK_SIZE + x;
				int gy = cy * CHUNK_SIZE + y;
				int gz = cz * CHUNK_SIZE + z;
				
				if(shouldRenderFace(world, gx, gy + 1, gz)) {
					UVRect uv = block->skin.top;
					glTexCoord2f(uv.u1, uv.v1);
					glVertex3f(gx, gy + 1, gz + 1);					
					glTexCoord2f(uv.u1, uv.v2);
					glVertex3f(gx, gy + 1, gz);
					glTexCoord2f(uv.u2, uv.v2);
					glVertex3f(gx + 1, gy + 1, gz);
					glTexCoord2f(uv.u2, uv.v1);
					glVertex3f(gx + 1, gy + 1, gz + 1);

				}
				if(shouldRenderFace(world, gx, gy - 1, gz)) {
					UVRect uv = block->skin.bottom;
					glTexCoord2f(uv.u1, uv.v1);
					glVertex3f(gx, gy, gz);
					glTexCoord2f(uv.u1, uv.v2);
					glVertex3f(gx, gy, gz + 1);
					glTexCoord2f(uv.u2, uv.v2);
					glVertex3f(gx + 1, gy, gz + 1);
					glTexCoord2f(uv.u2, uv.v1);
					glVertex3f(gx + 1, gy, gz);

				}
				
				UVRect uv = block->skin.side;
				if(shouldRenderFace(world, gx, gy, gz + 1)) {
					glTexCoord2f(uv.u1, uv.v1);
					glVertex3f(gx, gy, gz + 1);
					glTexCoord2f(uv.u1, uv.v2);
					glVertex3f(gx, gy + 1, gz + 1);
					glTexCoord2f(uv.u2, uv.v2);
					glVertex3f(gx + 1, gy + 1, gz + 1);
					glTexCoord2f(uv.u2, uv.v1);
					glVertex3f(gx + 1, gy, gz + 1);					
				}
				if(shouldRenderFace(world, gx, gy, gz - 1)) {
					glTexCoord2f(uv.u1, uv.v1);
					glVertex3f(gx + 1, gy, gz);				
					glTexCoord2f(uv.u1, uv.v2);
					glVertex3f(gx + 1, gy + 1, gz);
					glTexCoord2f(uv.u2, uv.v2);
					glVertex3f(gx, gy + 1, gz);
					glTexCoord2f(uv.u2, uv.v1);
					glVertex3f(gx, gy, gz);
				}
				if(shouldRenderFace(world, gx + 1, gy, gz)) {
					glTexCoord2f(uv.u1, uv.v1);
					glVertex3f(gx + 1, gy, gz + 1);
					glTexCoord2f(uv.u2, uv.v1);
					glVertex3f(gx + 1, gy, gz);
					glTexCoord2f(uv.u2, uv.v2);
					glVertex3f(gx + 1, gy + 1, gz);
					glTexCoord2f(uv.u1, uv.v2);
					glVertex3f(gx + 1, gy + 1, gz + 1);
				}
				if(shouldRenderFace(world, gx - 1, gy, gz)) {
					glTexCoord2f(uv.u1, uv.v1);
					glVertex3f(gx, gy, gz);
					glTexCoord2f(uv.u2, uv.v1);
					glVertex3f(gx, gy, gz + 1);
					glTexCoord2f(uv.u2, uv.v2);
					glVertex3f(gx, gy + 1, gz + 1);
					glTexCoord2f(uv.u1, uv.v2);
					glVertex3f(gx, gy + 1, gz);
				}
			}
		}
	}
	glEnd();

	glEndList();
	needsUpdate = false;
}

void Chunk::display() {
	if(listId != 0) glCallList(listId);
}

World::World(int& nowSlotNumber, GLuint& terrainAtlasID) : width(WORLD_CHUNKS_X * CHUNK_SIZE), height(WORLD_CHUNKS_Y * CHUNK_SIZE), depth(WORLD_CHUNKS_Z * CHUNK_SIZE), blockRegistry(nowSlotNumber, terrainAtlasID) {
	for(int x = 0; x < WORLD_CHUNKS_X; x ++) {
		for(int y = 0; y < WORLD_CHUNKS_Y; y ++) {
			for(int z = 0; z < WORLD_CHUNKS_Z; z ++) {
				chunks[x][y][z] = new Chunk();
				chunks[x][y][z]->init(x, y, z);
			}
		}
	}
}

World::~World() {
	for(int x = 0; x < WORLD_CHUNKS_X; x ++) {
		for(int y = 0; y < WORLD_CHUNKS_Y; y ++) {
			for(int z = 0; z < WORLD_CHUNKS_Z; z ++) {
				delete chunks[x][y][z];
			}
		}
	}
}

int World::getBlock(int x, int y, int z) {
	if(x < 0 || x >= width || y < 0 || y >= height || z < 0 || z >= depth) {
		return 0;
	}
	return chunks[x / CHUNK_SIZE][y / CHUNK_SIZE][z / CHUNK_SIZE]->blocks[x % CHUNK_SIZE][y % CHUNK_SIZE][z % CHUNK_SIZE];
}

void World::setBlock(int x, int y, int z, int id) {
	if(x < 0 || x >= width || y < 0 || y >= height || z < 0 || z >= depth){
		return;
	}
	
	int chX = x / CHUNK_SIZE;
	int chY = y / CHUNK_SIZE;
	int chZ = z / CHUNK_SIZE;
	int lx = x % CHUNK_SIZE;
	int ly = y % CHUNK_SIZE;
	int lz = z % CHUNK_SIZE;

	chunks[chX][chY][chZ]->blocks[lx][ly][lz] = id;
	chunks[chX][chY][chZ]->needsUpdate = true;

	if(lx == 0 && chX > 0) {
		chunks[chX - 1][chY][chZ]->needsUpdate = true;
	}
	if(lx == CHUNK_SIZE - 1 && chX < WORLD_CHUNKS_X - 1) {
		chunks[chX + 1][chY][chZ]->needsUpdate = true;
	}
	if(ly == 0 && chY > 0) {
		chunks[chX][chY - 1][chZ]->needsUpdate = true;
	}
	if(ly == CHUNK_SIZE - 1 && chY < WORLD_CHUNKS_Y - 1) {
		chunks[chX][chY + 1][chZ]->needsUpdate = true;
	}
	if(lz == 0 && chZ > 0) {
		chunks[chX][chY][chZ - 1]->needsUpdate = true;
	}
	if(lz == CHUNK_SIZE - 1 && chZ < WORLD_CHUNKS_Z - 1) {
		chunks[chX][chY][chZ + 1]->needsUpdate = true;
	}
}

void World::display(GLuint terrainAtlasID) {
	for (int x = 0; x < WORLD_CHUNKS_X; x++) {
		for (int y = 0; y < WORLD_CHUNKS_Y; y++) {
			for (int z = 0; z < WORLD_CHUNKS_Z; z++) {
				if (chunks[x][y][z]->needsUpdate) {
					chunks[x][y][z]->updateDisplayList(*this, terrainAtlasID);
				}
				chunks[x][y][z]->display();
			}
		}
	}
}

bool World::loadFromBMP(const char* path) {
	FILE* f = fopen(path, "rb");
	if (!f) return false;

	unsigned char fileHeader[14];
	unsigned char infoHeader[40];

	fread(fileHeader, 1, 14, f);
	fread(infoHeader, 1, 40, f);

	int bitCount = infoHeader[14] | (infoHeader[15] << 8);
	if (bitCount != 8) {
		fclose(f);
		return false;
	}

	fread(palette.data, 4, 256, f);
	hasPalette = true;

	for (int v = 0; v < 1024; v++) {
		for (int u = 0; u < 2048; u++) {
			unsigned char id;
			fread(&id, 1, 1, f);

			int x, y, z;
			pixelToWorld(u, v, x, y, z);
			setBlock(x, y, z, id);
		}
	}

	fclose(f);
	return true;
}

bool World::saveToBMP(const char* path) {
	if(!hasPalette) {
		createDefaultPalette();
	}

	FILE* f = fopen(path, "wb+");
	if(!f) {
		return false;
	}
	
	const int W = 2048;
	const int H = 1024;
	const int paletteSize = 256 * 4;
	const int pixelSize = W * H;
	const int offset = 14 + 40 + paletteSize;
	const int fileSize = offset + pixelSize;

	unsigned char fileHeader[14] = {
		'B','M',
		(unsigned char)(fileSize),
		(unsigned char)(fileSize >> 8),
		(unsigned char)(fileSize >> 16),
		(unsigned char)(fileSize >> 24),
		0,0,0,0,
		(unsigned char)(offset),
		(unsigned char)(offset >> 8),
		(unsigned char)(offset >> 16),
		(unsigned char)(offset >> 24)
	};
	fwrite(fileHeader, 1, 14, f);
	
	unsigned char infoHeader[40] = {
		40,0,0,0,
		(unsigned char)(W),
		(unsigned char)(W >> 8),
		(unsigned char)(W >> 16),
		(unsigned char)(W >> 24),
		(unsigned char)(H),
		(unsigned char)(H >> 8),
		(unsigned char)(H >> 16),
		(unsigned char)(H >> 24),
		1,0,
		8,0,
		0,0,0,0,
		0,0,0,0,
		0,0,0,0,
		0,0,0,0,
		0,0,0,0
	};
	fwrite(infoHeader, 1, 40, f);
	
	fwrite(palette.data, 4, 256, f);

	for(int v = 0; v < H; v++) {
		for(int u = 0; u < W; u++) {
			int x, y, z;
			pixelToWorld(u, v, x, y, z);
			unsigned char id = getBlock(x, y, z);
			fwrite(&id, 1, 1, f);
		}
	}

	fclose(f);
	return true;
}

void World::createDefaultPalette() {
	for(int i = 0; i < 256; i++) {
		palette.data[i][0] = i * 64 % 256;
		palette.data[i][1] = i * 15 % 256;
		palette.data[i][2] = i * 3 % 256;
		palette.data[i][3] = 0;
	}
	hasPalette = true;
}

void pixelToWorld(int u, int v, int& x, int& y, int& z) {
	int tx = u / 128;
	int ty = v / 128;

	y = ty * 16 + tx;
	x = u % 128;
	z = v % 128;
}

RaycastResult raycast(Player player, World& world, float maxDist = 5.0f) {
	RaycastResult result = { false, 0, 0, 0, 0, 0, 0 };
	
	Vector rayPos(player.pos.x, player.pos.y + 1.5f, player.pos.z);
	Vector rayDir = player.camera.coo(1.0f).unit();

	float step = 0.1f;
	for(float t = 0; t < maxDist; t += step) {
		Vector current = rayPos + rayDir * t;
		int ix = (int)floor(current.x);
		int iy = (int)floor(current.y);
		int iz = (int)floor(current.z);
		
		if(ix >= 0 && ix < world.width && iy >= 0 && iy < world.height && iz >= 0 && iz < world.depth) {
			int blockID = world.getBlock(ix, iy, iz);
			if(world.blockRegistry.get(blockID)->isDisplayed || world.blockRegistry.get(blockID)->isCollidable) {
				result.hit = true;
				result.x = ix; result.y = iy; result.z = iz;
				
				Vector prev = rayPos + rayDir * (t - step);
				result.faceX = (int)floor(prev.x);
				result.faceY = (int)floor(prev.y);
				result.faceZ = (int)floor(prev.z);
				return result;
			}
		}
	}
	return result;
}

